"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getState = getState;
exports.toggleBlock = toggleBlock;
exports.toggleLine = toggleLine;
exports.toggleHeading = toggleHeading;
exports.replaceSelection = replaceSelection;
exports.toggleTaskList = toggleTaskList;
exports.insertLink = insertLink;

/**
 * The state of CodeMirror at the given position.
 *
 * @param {object} cm  The CodeMirror instance
 * @param {number} pos  The cursor pos
 * @returns {object} The state
 */
function getState(cm, pos) {
  pos = pos || cm.getCursor('start');
  var stat = cm.getTokenAt(pos);

  if (!stat.type) {
    return {};
  }

  var types = stat.type.split(' ');
  var ret = {};
  var data;
  var text;

  for (var i = 0; i < types.length; i++) {
    data = types[i];

    if (data === 'strong') {
      ret.bold = true;
    } else if (data === 'variable-2') {
      text = cm.getLine(pos.line);

      if (/^\s*\d+\.\s/.test(text)) {
        ret['ordered-list'] = true;
      } else {
        ret['unordered-list'] = true;
      }

      if (/^\s*(\d+\.|\*|\-|\+)\s\[[\sx]\]\s+/.test(text)) {
        ret['task-list'] = true;
      }
    } else if (data === 'atom') {
      ret.quote = true;
    } else if (data === 'em') {
      ret.italic = true;
    } else if (data === 'quote') {
      ret.quote = true;
    } else if (data === 'strikethrough') {
      ret.strikethrough = true;
    } else if (data === 'comment') {
      ret.code = true;
    }
  }

  return ret;
}

function toggleBlock(cm, type, startChars, endChars) {
  endChars = typeof endChars === 'undefined' ? startChars : endChars;
  var stat = getState(cm);
  var text;
  var start = startChars;
  var end = endChars;
  var startPoint = cm.getCursor('start');
  var endPoint = cm.getCursor('end');

  if (endPoint.ch === 0 && startPoint.line < endPoint.line) {
    endPoint.line -= 1;
    endPoint.ch = cm.getLine(endPoint.line).length;
  }

  cm.operation(() => {
    if (stat[type]) {
      text = cm.getLine(startPoint.line);
      start = text.slice(0, startPoint.ch);
      end = text.slice(startPoint.ch);

      if (type === 'bold') {
        start = start.replace(/(\*\*|__)(?![\s\S]*(\*\*|__))/, '');
        end = end.replace(/(\*\*|__)/, '');
      } else if (type === 'italic') {
        start = start.replace(/(\*|_)(?![\s\S]*(\*|_))/, '');
        end = end.replace(/(\*|_)/, '');
      } else if (type === 'strikethrough') {
        start = start.replace(/(\*\*|~~)(?![\s\S]*(\*\*|~~))/, '');
        end = end.replace(/(\*\*|~~)/, '');
      }

      cm.replaceRange(start + end, {
        line: startPoint.line,
        ch: 0
      }, {
        line: startPoint.line,
        ch: 99999999999999
      });

      if (type === 'bold' || type === 'strikethrough') {
        startPoint.ch -= 2;

        if (startPoint !== endPoint) {
          endPoint.ch -= 2;
        }
      } else if (type === 'italic') {
        startPoint.ch -= 1;

        if (startPoint !== endPoint) {
          endPoint.ch -= 1;
        }
      }
    } else {
      text = cm.getSelection();

      if (type === 'bold') {
        text = text.split('**').join('');
        text = text.split('__').join('');
      } else if (type === 'italic') {
        text = text.split('*').join('');
        text = text.split('_').join('');
      } else if (type === 'strikethrough') {
        text = text.split('~~').join('');
      }

      cm.replaceSelection(start + text + end);
      startPoint.ch += startChars.length;
      endPoint.ch = startPoint.ch + text.length;
    }

    cm.setSelection(startPoint, endPoint);
  });
  cm.focus();
}

function toggleLine(cm, name) {
  const stat = getState(cm);
  const unorderedListBullet = cm.getOption('unorderedListBullet') || '*';
  var startPoint = cm.getCursor('start');
  var endPoint = cm.getCursor('end');
  var repl = {
    quote: /^(\s*)>\s+/,
    'unordered-list': /^(\s*)(\*|\-|\+)\s+/,
    'ordered-list': /^(\s*)\d+\.\s+/,
    'task-list': /^(\s*)(\d+\.|\*|\-|\+)\s\[[\sx]\]\s+/
  };
  var map = {
    quote: '> ',
    'unordered-list': `${unorderedListBullet} `,
    'ordered-list': '1. ',
    'task-list': `${unorderedListBullet} [ ] `
  };

  for (var i = startPoint.line; i <= endPoint.line; i++) {
    var text = cm.getLine(i);

    if (stat[name]) {
      text = text.replace(repl[name], '$1');
    } else {
      text = map[name] + text;
    }

    cm.replaceRange(text, {
      line: i,
      ch: 0
    }, {
      line: i,
      ch: 99999999999999
    });
  }

  cm.focus();
}

function toggleHeading(cm, direction, size) {
  var startPoint = cm.getCursor('start');
  var endPoint = cm.getCursor('end');

  for (var i = startPoint.line; i <= endPoint.line; i++) {
    var text = cm.getLine(i);
    var currHeadingLevel = text.search(/[^#]/);

    if (direction) {
      if (currHeadingLevel <= 0) {
        if (direction === 'bigger') {
          text = '###### ' + text;
        } else {
          text = '# ' + text;
        }
      } else if (currHeadingLevel === 6 && direction === 'smaller') {
        text = text.substr(7);
      } else if (currHeadingLevel === 1 && direction === 'bigger') {
        text = text.substr(2);
      } else {
        if (direction === 'bigger') {
          text = text.substr(1);
        } else {
          text = '#' + text;
        }
      }
    } else {
      if (size === 1) {
        if (currHeadingLevel <= 0) {
          text = '# ' + text;
        } else if (currHeadingLevel === size) {
          text = text.substr(currHeadingLevel + 1);
        } else {
          text = '# ' + text.substr(currHeadingLevel + 1);
        }
      } else if (size === 2) {
        if (currHeadingLevel <= 0) {
          text = '## ' + text;
        } else if (currHeadingLevel === size) {
          text = text.substr(currHeadingLevel + 1);
        } else {
          text = '## ' + text.substr(currHeadingLevel + 1);
        }
      } else if (size === 3) {
        if (currHeadingLevel <= 0) {
          text = '### ' + text;
        } else if (currHeadingLevel === size) {
          text = text.substr(currHeadingLevel + 1);
        } else {
          text = '### ' + text.substr(currHeadingLevel + 1);
        }
      } else {
        if (currHeadingLevel <= 0) {
          text = '#### ' + text;
        } else if (currHeadingLevel === size) {
          text = text.substr(currHeadingLevel + 1);
        } else {
          text = '#### ' + text.substr(currHeadingLevel + 1);
        }
      }
    }

    cm.replaceRange(text, {
      line: i,
      ch: 0
    }, {
      line: i,
      ch: 99999999999999
    });
  }

  cm.focus();
}

function replaceSelection(cm, active, startEnd) {
  var text;
  var start = startEnd[0];
  var end = startEnd[1];
  var startPoint = cm.getCursor('start');
  var endPoint = cm.getCursor('end');

  if (active) {
    text = cm.getLine(startPoint.line);
    start = text.slice(0, startPoint.ch);
    end = text.slice(startPoint.ch);
    cm.replaceRange(start + end, {
      line: startPoint.line,
      ch: 0
    });
  } else {
    text = cm.getSelection();
    cm.replaceSelection(start + text + end);
    startPoint.ch += start.length;

    if (startPoint !== endPoint) {
      endPoint.ch += start.length;
    }
  }

  cm.setSelection(startPoint, endPoint);
  cm.focus();
}

function toggleTaskList(cm, lineNum, checked) {
  const doc = cm.getDoc();
  const line = doc.getLine(lineNum - 1);

  if (line) {
    const from = {
      line: lineNum - 1,
      ch: 0
    };
    const to = {
      line: lineNum - 1,
      ch: line.length
    };
    const todoRegex = /^(\s*)(\d*[-+*\.])\s\[[x ]\]/;
    const match = line.match(todoRegex);

    if (match) {
      const [, indent, bullet] = match;
      const replaceWith = `${indent}${bullet} [${checked ? 'x' : ' '}]`;
      const newLine = line.replace(todoRegex, replaceWith);
      doc.replaceRange(newLine, from, to);
      return true;
    }
  } else {
    return false;
  }
}

function insertLink(cm) {
  const stat = getState(cm);
  const text = cm.getSelection();
  const moveTo = Object.assign({}, cm.getCursor('start'));

  if (/^https?:\/\//.test(text)) {
    cm.replaceSelection(`[](${text})`);
    moveTo.ch += 1;
  } else if (text.length > 0) {
    cm.replaceSelection(`[${text}]()`);
    moveTo.ch += text.length + 3;
  } else {
    replaceSelection(cm, stat.link, ['[', ']()']);
    return;
  }

  cm.setSelection(moveTo, moveTo);
  cm.focus();
}