"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LinkType = exports.NextMaybe = exports.TableType = void 0;

var CodeMirror = _interopRequireWildcard(require("codemirror"));

require("codemirror/mode/markdown/markdown");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

// CodeMirror, copyright (c) by laobubu
// Distributed under an MIT license: http://codemirror.net/LICENSE
//
// This is a patch to markdown mode. Supports lots of new features
//
const TableType = {
  NONE: 0,
  SIMPLE: 1,
  //   table | column
  NORMAL: 2 // | table | column |

};
exports.TableType = TableType;
const NextMaybe = {
  NONE: 0,
  FRONT_MATTER: 1,
  // only appears once, for each Doc
  FRONT_MATTER_END: 2 // endline of front_matter not reached

};
exports.NextMaybe = NextMaybe;
const LinkType = {
  NONE: 0,
  BARELINK: 1,
  // [link]
  FOOTREF: 2,
  // [^ref]
  NORMAL: 3,
  // [text](url) or [text][doc]
  FOOTNOTE: 4,
  // [footnote]:
  MAYBE_FOOTNOTE_URL: 5,
  // things after colon
  BARELINK2: 6,
  // [some-name][]  except latter []
  FOOTREF2: 7 // [text][doc]  the [doc] part

};
/**
 * Markdown Extension Tokens
 *
 * - `$` Maybe a LaTeX
 * - `|` Maybe a Table Col Separator
 */

exports.LinkType = LinkType;
const tokenBreakRE = /[^\\][$|]/;
const listRE = /^(?:[*\-+]|^[0-9]+([.)]))\s+/;
const urlRE = /^((?:(?:aaas?|about|acap|adiumxtra|af[ps]|aim|apt|attachment|aw|beshare|bitcoin|bolo|callto|cap|chrome(?:-extension)?|cid|coap|com-eventbrite-attendee|content|crid|cvs|data|dav|dict|dlna-(?:playcontainer|playsingle)|dns|doi|dtn|dvb|ed2k|facetime|feed|file|finger|fish|ftp|geo|gg|git|gizmoproject|go|gopher|gtalk|h323|hcp|https?|iax|icap|icon|im|imap|info|inkdrop|inkdrop-file|ipn|ipp|irc[6s]?|iris(?:\.beep|\.lwz|\.xpc|\.xpcs)?|itms|jar|javascript|jms|keyparc|lastfm|ldaps?|magnet|mailto|maps|market|message|mid|mms|ms-help|msnim|msrps?|mtqp|mumble|mupdate|mvn|news|nfs|nih?|nntp|notes|oid|opaquelocktoken|palm|paparazzi|platform|pop|pres|proxy|psyc|query|res(?:ource)?|rmi|rsync|rtmp|rtsp|secondlife|service|session|sftp|sgn|shttp|sieve|sips?|skype|sm[bs]|snmp|soap\.beeps?|soldat|spotify|ssh|steam|svn|tag|teamspeak|tel(?:net)?|tftp|things|thismessage|tip|tn3270|tv|udp|unreal|urn|ut2004|vemmi|ventrilo|view-source|webcal|wss?|wtai|wyciwyg|xcon(?:-userid)?|xfire|xmlrpc\.beeps?|xmpp|xri|ymsgr|z39\.50[rs]?):(?:\/{1,3}|[a-z0-9%])|www\d{0,3}[.]|[a-z0-9.\-]+[.][a-z]{2,4}\/)(?:[^\s()<>]|\([^\s()<>]*\))+(?:\([^\s()<>]*\)|[^\s`*!()\[\]{};:'".,<>?«»“”‘’]))/i; // from CodeMirror/mode/gfm

const emailRE = /^(([^<>()[\]\\.,;:\s@\"]+(\.[^<>()[\]\\.,;:\s@\"]+)*)|(\".+\"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
const url2RE = /^\.{0,2}\/[^\>\s]+/;
const simpleTableRE = /^\s*[^\|].*?\|.*[^|]\s*$/;
const simpleTableLooseRE = /^\s*[^\|].*\|/; // unfinished | row

const normalTableRE = /^\s*\|[^\|]+\|.+\|\s*$/;
const normalTableLooseRE = /^\s*\|/; // | unfinished row

const linkStyle = {
  [LinkType.BARELINK]: 'link-barelink',
  [LinkType.BARELINK2]: 'link-barelink2',
  [LinkType.FOOTREF]: 'link-barelink link-footref',
  [LinkType.FOOTNOTE]: 'link-footnote',
  [LinkType.FOOTREF2]: 'link-footref2'
};

function resetTable(state) {
  state.table = TableType.NONE;
  state.tableColumns = [];
  state.tableID = null;
  state.tableCol = state.tableRow = 0;
}

const listInQuoteRE = /^\s+((\d+[).]|[-*+])\s+)?/;
const defaultTokenTypeOverrides = {
  code: 'inline-code'
};
CodeMirror.defineMode('gfm', function (cmCfg, modeCfgUser) {
  var modeCfg = {
    front_matter: true,
    math: true,
    table: true,
    fencedCodeBlockHighlighting: true,
    name: 'markdown',
    highlightFormatting: true,
    taskLists: true,
    strikethrough: true,
    emoji: true,

    /** @see defaultTokenTypeOverrides */
    tokenTypeOverrides: defaultTokenTypeOverrides
  };
  Object.assign(modeCfg, modeCfgUser);

  if (modeCfg.tokenTypeOverrides !== defaultTokenTypeOverrides) {
    modeCfg.tokenTypeOverrides = Object.assign({}, defaultTokenTypeOverrides, modeCfg.tokenTypeOverrides);
  }

  modeCfg['name'] = 'markdown';
  /** functions from CodeMirror Markdown mode closure. Only for status checking */

  var rawClosure = {
    htmlBlock: null
  };
  var rawMode = CodeMirror.getMode(cmCfg, modeCfg);
  var newMode = Object.assign({}, rawMode);

  newMode.startState = function () {
    var ans = rawMode.startState();
    resetTable(ans);
    ans.tokenOverride = null;
    ans.innerExitChecker = null;
    ans.innerMode = null;
    ans.linkType = LinkType.NONE;
    ans.nextMaybe = modeCfg.front_matter ? NextMaybe.FRONT_MATTER : NextMaybe.NONE;
    ans.nextState = null;
    ans.nextStyle = null;
    ans.nextPos = null;
    return ans;
  };

  newMode.copyState = function (s) {
    var ans = rawMode.copyState(s);
    const keys = ['linkType', 'nextMaybe', 'table', 'tableID', 'tableCol', 'tableRow', 'tokenOverride', 'innerMode', 'innerStyle', 'innerExitChecker', 'nextPos', 'nextState', 'nextStyle'];

    for (const key of keys) ans[key] = s[key];

    ans.tableColumns = s.tableColumns.slice(0);
    if (s.innerMode) ans.innerState = CodeMirror.copyState(s.innerMode, s.innerState);
    return ans;
  };

  newMode.blankLine = function (state) {
    var ans;
    var innerMode = state.innerMode;

    if (innerMode) {
      if (innerMode.blankLine) ans = innerMode.blankLine(state.innerState);
    } else {
      ans = rawMode.blankLine(state);
    }

    if (!ans) ans = '';

    if (state.code === -1) {
      ans += ' line-gfm-codeblock line-background-gfm-codeblock-bg';
    }

    resetTable(state);
    return ans.trim() || null;
  };

  newMode.indent = function (state, _textAfter) {
    var mode = state.innerMode || rawMode;
    var f = mode.indent;
    if (typeof f === 'function') return f.apply(mode, arguments);
    return CodeMirror.Pass;
  };

  newMode.innerMode = function (state) {
    if (state.innerMode) return {
      mode: state.innerMode,
      state: state.innerState
    };
    return rawMode.innerMode(state);
  };

  newMode.token = function (stream, state) {
    if (state.tokenOverride) return state.tokenOverride(stream, state);

    if (state.nextMaybe === NextMaybe.FRONT_MATTER) {
      // Only appears once for each Doc
      if (stream.string === '---') {
        state.nextMaybe = NextMaybe.FRONT_MATTER_END;
        return enterMode(stream, state, 'yaml', {
          style: 'frontmatter',
          fallbackMode: () => createDummyMode('---'),
          exitChecker: function (stream, state) {
            if (stream.string === '---') {
              // found the endline of front_matter
              state.nextMaybe = NextMaybe.NONE;
              return {
                endPos: 3
              };
            } else {
              return null;
            }
          }
        });
      } else {
        state.nextMaybe = NextMaybe.NONE;
      }
    }

    const wasInHTML = state.f === rawClosure.htmlBlock;
    const wasInCodeFence = state.code === -1;
    const bol = stream.start === 0;
    const wasLinkText = state.linkText;
    const wasLinkHref = state.linkHref;
    let inMarkdown = !(wasInCodeFence || wasInHTML);
    let inMarkdownInline = inMarkdown && !(state.code || state.indentedCode || state.linkHref);
    var ans = '';
    var tmp;

    if (inMarkdown) {
      // now implement some extra features that require higher priority than CodeMirror's markdown
      //#region Math
      if (modeCfg.math && inMarkdownInline && (tmp = stream.match(/^\${1,2}/, false))) {
        const endTag = tmp[0];
        const mathLevel = endTag.length;

        if (mathLevel === 2 || stream.string.slice(stream.pos).match(/[^\\]\$/)) {
          // $$ may span lines, $ must be paired
          const texMode = CodeMirror.getMode(cmCfg, {
            name: 'stex'
          });
          const noTexMode = texMode['name'] !== 'stex';
          ans += enterMode(stream, state, texMode, {
            style: 'math',
            skipFirstToken: noTexMode,
            fallbackMode: () => createDummyMode(endTag),
            exitChecker: createSimpleInnerModeExitChecker(endTag, {
              style: 'formatting formatting-math formatting-math-end math-' + mathLevel
            })
          });
          if (noTexMode) stream.pos += tmp[0].length;
          ans += ' formatting formatting-math formatting-math-begin math-' + mathLevel;
          return ans;
        }
      } //#endregion
      //#region Extra markdown inline extenson


      if (inMarkdownInline) {
        // transform unformatted URL into link
        if (!state.linkType && (stream.match(urlRE) || stream.match(emailRE))) {
          return 'url';
        }
      } //#endregion

    } // now enter markdown


    if (state.nextState) {
      stream.pos = state.nextPos;
      ans += ' ' + (state.nextStyle || '');
      Object.assign(state, state.nextState);
      state.nextState = null;
      state.nextStyle = null;
      state.nextPos = null;
    } else {
      ans += ' ' + (rawMode.token(stream, state) || '');
    }
    /** Try to capture some internal functions from CodeMirror Markdown mode closure! */


    if (!rawClosure.htmlBlock && state.htmlState) rawClosure.htmlBlock = state.f;
    const inHTML = state.f === rawClosure.htmlBlock;
    const inCodeFence = state.code === -1;
    inMarkdown = inMarkdown && !(inHTML || inCodeFence);
    inMarkdownInline = inMarkdownInline && inMarkdown && !(state.code || state.indentedCode || state.linkHref); // If find a markdown extension token (which is not escaped),
    // break current parsed string into two parts and the first char of next part is the markdown extension token

    if (inMarkdownInline && (tmp = stream.current().match(tokenBreakRE))) {
      stream.pos = stream.start + tmp.index + 1; // rewind
    }

    var current = stream.current();

    if (inHTML != wasInHTML) {
      if (inHTML) {
        ans += ' html-begin';
        rawClosure.htmlBlock = state.f;
      } else {
        ans += ' html-end';
      }
    }

    if (wasInCodeFence || inCodeFence || state.indentedCode) {
      if (!state.localMode || !wasInCodeFence) ans = ans.replace('inline-code', '');
      ans += ' line-gfm-codeblock line-background-gfm-codeblock-bg';
      state.codeblock = true;

      if (inCodeFence !== wasInCodeFence) {
        if (!inCodeFence) ans += ' line-gfm-codeblock-end line-background-gfm-codeblock-end-bg';else if (!wasInCodeFence) ans += ' line-gfm-codeblock-begin line-background-gfm-codeblock-begin-bg';
      }
    } else {
      state.codeblock = false;
    }

    if (inMarkdown) {
      let tableType = state.table; //#region [Table] Reset

      if (bol && tableType) {
        const rowRE = tableType == TableType.SIMPLE ? simpleTableLooseRE : normalTableLooseRE;

        if (rowRE.test(stream.string)) {
          // still in table
          state.tableCol = 0;
          state.tableRow++;
        } else {
          // end of a table
          resetTable(state);
        }
      } //#endregion
      //#region Header, indentedCode, quote


      if (bol && state.header) {
        if (/^(?:---+|===+)\s*$/.test(stream.string) && state.prevLine && state.prevLine.header) {
          ans += ' line-header-line line-header-line-' + state.header;
        } else {
          ans += ' line-header line-header-' + state.header;
        }
      }

      if (state.indentedCode) {
        ans += ' indented-code';
      }

      if (state.quote) {
        // mess up as less as possible
        if (stream.eol()) {
          ans += ' line-quote line-quote-' + state.quote;
          if (!/^ {0,3}\>/.test(stream.string)) ans += ' line-quote-lazy'; // ">" is omitted
        }

        if (bol && (tmp = current.match(/^\s+/))) {
          stream.pos = tmp[0].length; // rewind

          ans += ' indent-in-quote';
          return ans.trim();
        } // make indentation (and potential list bullet) monospaced


        if (/^>\s+$/.test(current) && stream.peek() != '>') {
          stream.pos = stream.start + 1; // rewind!

          current = '>';

          state.tokenOverride = (stream, state) => {
            stream.match(listInQuoteRE);
            state.tokenOverride = null;
            return 'indent-in-quote line-quote line-quote-' + state.quote;
          };
        }
      } //#endregion
      //#region List


      const maxNonCodeIndentation = (state.listStack[state.listStack.length - 1] || 0) + 3;
      const tokenIsIndent = bol && /^\s+$/.test(current) && (state.list !== false || stream.indentation() <= maxNonCodeIndentation);
      const tokenIsListBullet = state.list && /formatting-list/.test(ans);

      if (tokenIsListBullet || tokenIsIndent && (state.list !== false || stream.match(listRE, false))) {
        let listLevel = state.listStack && state.listStack.length || 0;

        if (tokenIsIndent) {
          if (stream.match(listRE, false)) {
            // next token is 1. 2. or bullet
            if (state.list === false) listLevel++;
          } else {
            while (listLevel > 0 && stream.pos < state.listStack[listLevel - 1]) {
              listLevel--; // find the real indent level
            }

            if (!listLevel) {
              // not even a list
              return ans.trim() || null;
            }

            ans += ` line-list-line-nobullet line-list-line line-list-line-${listLevel}`;
          }

          ans += ` list-indent list-indent-${listLevel}`;
        } else if (tokenIsListBullet) {
          // no space before bullet!
          ans += ` line-list-line line-list-line-${listLevel}`;
        }
      } //#endregion
      //#region Link, BareLink, Footnote etc


      if (wasLinkText !== state.linkText) {
        if (!wasLinkText) {
          // entering a link
          tmp = stream.match(/^([^\]]+)\](\(| ?\[|\:)?/, false);

          if (!tmp) {
            // maybe met a line-break in link text?
            state.linkType = LinkType.BARELINK;
          } else if (!tmp[2]) {
            // barelink
            if (tmp[1].charAt(0) === '^') {
              state.linkType = LinkType.FOOTREF;
            } else {
              state.linkType = LinkType.BARELINK;
            }
          } else if (tmp[2] === ':') {
            // footnote
            state.linkType = LinkType.FOOTNOTE;
          } else if ((tmp[2] === '[' || tmp[2] === ' [') && stream.string.charAt(stream.pos + tmp[0].length) === ']') {
            // [barelink2][]
            state.linkType = LinkType.BARELINK2;
          } else {
            state.linkType = LinkType.NORMAL;
          }
        } else {
          // leaving a link
          if (state.linkType in linkStyle) ans += ' ' + linkStyle[state.linkType];

          if (state.linkType === LinkType.FOOTNOTE) {
            state.linkType = LinkType.MAYBE_FOOTNOTE_URL;
          } else {
            state.linkType = LinkType.NONE;
          }
        }
      }

      if (wasLinkHref !== state.linkHref) {
        if (!wasLinkHref) {
          // [link][doc] the [doc] part
          if (current === '[' && stream.peek() !== ']') {
            state.linkType = LinkType.FOOTREF2;
          }
        } else if (state.linkType) {
          // leaving a Href
          ans += ' ' + linkStyle[state.linkType];
          state.linkType = LinkType.NONE;
        }
      }

      if (state.linkType !== LinkType.NONE) {
        if (state.linkType in linkStyle) ans += ' ' + linkStyle[state.linkType];

        if (state.linkType === LinkType.MAYBE_FOOTNOTE_URL) {
          if (!/^(?:\]\:)?\s*$/.test(current)) {
            // not spaces
            if (urlRE.test(current) || url2RE.test(current)) ans += ' footnote-url';else ans = ans.replace('string url', ''); // since then, can't be url anymore

            state.linkType = LinkType.NONE;
          }
        }
      } //#endregion
      //#region start of an escaped char


      if (/formatting-escape/.test(ans) && current.length > 1) {
        // CodeMirror merge backslash and escaped char into one token, which is not good
        // Use tokenOverride to separate them
        const escapedLength = current.length - 1;
        const escapedCharStyle = ans.replace('formatting-escape', 'escape') + ' escape-char';

        state.tokenOverride = (stream, state) => {
          stream.pos += escapedLength;
          state.tokenOverride = null;
          return escapedCharStyle.trim();
        };

        ans += ' escape-backslash';
        stream.pos -= escapedLength;
        return ans;
      } //#endregion
      //#region [Table] Creating Table and style Table Separators


      if (!ans.trim() && modeCfg.table) {
        // string is unformatted
        let isTableSep = false;

        if (current.charAt(0) === '|') {
          // is "|xxxxxx", separate "|" and "xxxxxx"
          stream.pos = stream.start + 1; // rewind to end of "|"

          current = '|';
          isTableSep = true;
        }

        if (isTableSep) {
          // if not inside a table, try to construct one
          if (!tableType) {
            // check 1: current line meet the table format
            if (simpleTableRE.test(stream.string)) tableType = 1;
            /* SIMPLE */
            else if (normalTableRE.test(stream.string)) tableType = TableType.NORMAL; // check 2: check every column's alignment style

            let rowStyles;

            if (tableType) {
              let nextLine = stream.lookAhead(1);

              if (tableType === TableType.NORMAL) {
                if (!normalTableRE.test(nextLine)) {
                  tableType = TableType.NONE;
                } else {
                  // remove leading / tailing pipe char
                  nextLine = nextLine.replace(/^\s*\|/, '').replace(/\|\s*$/, '');
                }
              } else if (tableType === TableType.SIMPLE) {
                if (!simpleTableRE.test(nextLine)) {
                  tableType = TableType.NONE;
                }
              }

              if (tableType) {
                rowStyles = nextLine.split('|');

                for (let i = 0; i < rowStyles.length; i++) {
                  let row = rowStyles[i];
                  if (/^\s*--+\s*:\s*$/.test(row)) row = 'right';else if (/^\s*:\s*--+\s*$/.test(row)) row = 'left';else if (/^\s*:\s*--+\s*:\s*$/.test(row)) row = 'center';else if (/^\s*--+\s*$/.test(row)) row = 'default';else {
                    // ouch, can't be a table
                    tableType = TableType.NONE;
                    break;
                  }
                  rowStyles[i] = row;
                }
              }
            } // step 3: made it


            if (tableType) {
              // successfully made one
              state.table = tableType;
              state.tableColumns = rowStyles;
              state.tableID = 'T' + stream.lineOracle.line;
              state.tableRow = state.tableCol = 0;
            }
          } // then


          if (tableType) {
            const colUbound = state.tableColumns.length - 1;

            if (tableType === TableType.NORMAL && (state.tableCol === 0 && /^\s*\|$/.test(stream.string.slice(0, stream.pos)) || stream.match(/^\s*$/, false))) {
              ans += ` table-sep table-sep-dummy`;
            } else if (state.tableCol < colUbound) {
              const row = state.tableRow;
              const col = state.tableCol++;

              if (col == 0) {
                ans += ` line-table_${state.tableID} line-table-${tableType} line-table-row line-table-row-${row}`;
              }

              ans += ` table-sep table-sep-${col}`;
            }
          }
        }
      } //#endregion


      if (tableType && state.tableRow === 1) {
        // fix a stupid problem:    :------: is not emoji
        if (/emoji/.test(ans)) ans = '';
      } //#region HTML Block
      //
      // See https://github.github.com/gfm/#html-blocks type3-5


      if (inMarkdownInline && current === '<') {
        let endTag = null;
        if (stream.match(/^\![A-Z]+/)) endTag = '>';else if (stream.match('?')) endTag = '?>';else if (stream.match('![CDATA[')) endTag = ']]>';

        if (endTag != null) {
          return enterMode(stream, state, null, {
            endTag,
            style: (ans + ' comment cdata-html').trim()
          });
        }
      } //#endregion

    }

    return ans.trim() || null;
  };

  function modeOverride(stream, state) {
    const exit = state.innerExitChecker(stream, state);
    const extraStyle = state.innerStyle;
    let ans = (!exit || !exit.skipInnerMode) && state.innerMode.token(stream, state.innerState) || '';
    if (extraStyle) ans += ' ' + extraStyle;

    if (exit) {
      if (exit.style) ans += ' ' + exit.style;
      if (exit.endPos) stream.pos = exit.endPos;
      state.innerExitChecker = null;
      state.innerMode = null;
      state.innerStyle = null;
      state.innerState = null;
      state.tokenOverride = null;
    }

    return ans.trim() || null;
  }
  /**
   * advance Markdown tokenizing stream
   *
   * @returns true if success, then `state.nextState` & `state.nextStyle` will be set
   */


  function _advanceMarkdown(stream, state) {
    if (stream.eol() || state.nextState) return false;
    var oldStart = stream.start;
    var oldPos = stream.pos;
    stream.start = oldPos;
    var newState = Object.assign({}, state);
    var newStyle = rawMode.token(stream, newState);
    state.nextPos = stream.pos;
    state.nextState = newState;
    state.nextStyle = newStyle; // console.log("ADVANCED!", oldStart, oldPos, stream.start, stream.pos)
    // console.log("ADV", newStyle, newState)

    stream.start = oldStart;
    stream.pos = oldPos;
    return true;
  }

  function createDummyMode(endTag) {
    return {
      token(stream) {
        var endTagSince = stream.string.indexOf(endTag, stream.start);
        if (endTagSince === -1) stream.skipToEnd(); // endTag not in this line
        else if (endTagSince === 0) stream.pos += endTag.length; // current token is endTag
          else {
              stream.pos = endTagSince;
              if (stream.string.charAt(endTagSince - 1) === '\\') stream.pos++;
            }
        return null;
      }

    };
  }

  function createSimpleInnerModeExitChecker(endTag, retInfo) {
    if (!retInfo) retInfo = {};
    return function (stream, _state) {
      if (stream.string.substr(stream.start, endTag.length) === endTag) {
        retInfo.endPos = stream.start + endTag.length;
        return retInfo;
      }

      return null;
    };
  }
  /**
   * switch to another mode
   *
   * After entering a mode, you can then set `innerExitStyle` and `innerState` of `state`
   *
   * @returns if `skipFirstToken` not set, returns `innerMode.token(stream, innerState)`, meanwhile, stream advances
   */


  function enterMode(stream, state, mode, opt) {
    if (typeof mode === 'string') mode = CodeMirror.getMode(cmCfg, mode);

    if (!mode || mode['name'] === 'null') {
      if ('endTag' in opt) mode = createDummyMode(opt.endTag);else mode = typeof opt.fallbackMode === 'function' && opt.fallbackMode();
      if (!mode) throw new Error('no mode');
    }

    state.innerExitChecker = 'endTag' in opt ? createSimpleInnerModeExitChecker(opt.endTag) : opt.exitChecker;
    state.innerStyle = opt.style;
    state.innerMode = mode;
    state.tokenOverride = modeOverride;
    state.innerState = CodeMirror.startState(mode);
    var ans = opt.style || '';

    if (!opt.skipFirstToken) {
      ans += ' ' + mode.token(stream, state.innerState);
    }

    return ans.trim();
  }

  return newMode;
}, 'markdown');
CodeMirror.defineMIME('text/x-gfm', 'gfm');