// HyperMD, copyright (c) by laobubu
// Distributed under an MIT license: http://laobubu.net/HyperMD/LICENSE
//
// powerful keymap for HyperMD and Markdown modes
//
import * as CodeMirror from 'codemirror'
import { cmpPos } from 'codemirror'
import repeat from 'repeat-string'
import { moveToNextTableCell, moveToPreviousTableCell } from './table'
import { getGfmState } from './utils'

const ListRE = /^(\s*)([*+-]\s|(\d+)([.)]))(\s*)/

function getLineSafely(cm, lineNo) {
  return cm.getLine(lineNo).replace(/\u00A0/g, ' ')
}

function killIndent(cm, lineNo, spaces) {
  if (!spaces || spaces < 0) return
  const oldSpaces = /^ */.exec(getLineSafely(cm, lineNo))[0].length
  if (oldSpaces < spaces) spaces = oldSpaces
  if (spaces > 0)
    cm.replaceRange('', { line: lineNo, ch: 0 }, { line: lineNo, ch: spaces })
}

/** unindent or move cursor into prev table cell */
export function shiftTab(cm) {
  const enableTableHelpers = cm.getOption('tableHelpers')
  var selections = cm.listSelections()
  for (let i = 0; i < selections.length; i++) {
    var range = selections[i]
    var left = range.head
    var right = range.anchor
    const rangeEmpty = range.empty()
    if (!rangeEmpty && cmpPos(left, right) > 0) [right, left] = [left, right]
    else if (right === left) {
      right = range.anchor = { ch: left.ch, line: left.line }
    }

    if (enableTableHelpers) {
      const sel = moveToPreviousTableCell(cm, range)
      if (sel) {
        cm.setSelection(sel.anchor, sel.head)
        return
      }
    }

    const eolState = getGfmState(cm.getStateAfter(left.line))
    if (eolState.listStack.length > 0) {
      let lineNo = left.line
      while (!ListRE.test(getLineSafely(cm, lineNo))) {
        // beginning line has no bullet? go up
        lineNo--
        const isList =
          getGfmState(cm.getStateAfter(lineNo)).listStack.length > 0
        if (!isList) {
          lineNo++
          break
        }
      }
      const lastLine = cm.lastLine()
      let tmp
      for (
        ;
        lineNo <= right.line && (tmp = ListRE.exec(getLineSafely(cm, lineNo)));
        lineNo++
      ) {
        const listStack = getGfmState(cm.getStateAfter(lineNo)).listStack
        const listLevel = listStack.length
        let spaces = 0
        if (listLevel == 1) {
          // maybe user wants to trimLeft?
          spaces = tmp[1].length
        } else {
          // make bullets right-aligned
          spaces = listStack[listLevel - 1] - (listStack[listLevel - 2] || 0)
        }
        killIndent(cm, lineNo, spaces)
        // if current list item is multi-line...
        while (++lineNo <= lastLine) {
          if (
            /*corrupted */ getGfmState(cm.getStateAfter(lineNo)).listStack
              .length !== listLevel
          ) {
            lineNo = Infinity
            break
          }
          if (/*has bullet*/ ListRE.test(getLineSafely(cm, lineNo))) {
            lineNo--
            break
          }
          killIndent(cm, lineNo, spaces)
        }
      }
      return
    }
  }
  cm.execCommand('indentLess')
}

/**
 * 1. for tables, move cursor into next table cell, and maybe insert a cell
 * 2.
 */
export function tab(cm) {
  const enableTableHelpers = cm.getOption('tableHelpers')
  var selections = [...cm.listSelections()]
  var beforeCur = []
  var afterCur = []
  var addIndentTo = {} // {lineNo: stringIndent}
  /** indicate previous 4 variable changed or not */
  var flag0 = false,
    flag1 = false,
    flag2 = false
  function setBeforeCur(i, text) {
    beforeCur[i] = text
    if (text) flag1 = true
  }
  function setAfterCur(i, text) {
    afterCur[i] = text
    if (text) flag2 = true
  }
  for (let i = 0; i < selections.length; i++) {
    beforeCur[i] = afterCur[i] = ''
    var range = selections[i]
    var left = Object.assign({}, range.head)
    var right = Object.assign({}, range.anchor)
    const rangeEmpty = range.empty()
    if (!rangeEmpty && cmpPos(left, right) > 0) [right, left] = [left, right]
    else if (right === left) {
      right = range.anchor = { ch: left.ch, line: left.line }
    }
    const eolState = getGfmState(cm.getStateAfter(left.line))
    if (eolState.table && enableTableHelpers) {
      // yeah, we are inside a table
      flag0 = true // cursor will move
      const result = moveToNextTableCell(cm, range)
      result.beforeCur && setBeforeCur(i, result.beforeCur)
      result.afterCur && setAfterCur(i, result.afterCur)
      if (result.moveTo) {
        selections[i] = result.moveTo
      }
    } else if (eolState.listStack.length > 0) {
      // add indent to current line
      let lineNo = left.line
      let tmp // ["  * ", "  ", "* "]
      while (!(tmp = ListRE.exec(getLineSafely(cm, lineNo)))) {
        // beginning line has no bullet? go up
        lineNo--
        const isList =
          getGfmState(cm.getStateAfter(lineNo)).listStack.length > 0
        if (!isList) {
          lineNo++
          break
        }
      }
      const firstLine = cm.firstLine()
      const lastLine = cm.lastLine()
      for (
        ;
        lineNo <= right.line && (tmp = ListRE.exec(getLineSafely(cm, lineNo)));
        lineNo++
      ) {
        const eolState = getGfmState(cm.getStateAfter(lineNo))
        const listStack = eolState.listStack
        const listStackOfPrevLine = getGfmState(cm.getStateAfter(lineNo - 1))
          .listStack
        const listLevel = listStack.length
        let spaces = ''
        // avoid uncontinuous list levels
        if (lineNo > firstLine && listLevel <= listStackOfPrevLine.length) {
          if (listLevel == listStackOfPrevLine.length) {
            // tmp[1] is existed leading spaces
            // listStackOfPrevLine[listLevel-1] is desired indentation
            spaces = repeat(
              ' ',
              listStackOfPrevLine[listLevel - 1] - tmp[1].length
            )
          } else {
            // make bullets right-aligned
            // tmp[0].length is end pos of current bullet
            spaces = repeat(' ', listStackOfPrevLine[listLevel] - tmp[0].length)
          }
        }
        addIndentTo[lineNo] = spaces
        // if current list item is multi-line...
        while (++lineNo <= lastLine) {
          if (
            /*corrupted */ getGfmState(cm.getStateAfter(lineNo)).listStack
              .length !== listLevel
          ) {
            lineNo = Infinity
            break
          }
          if (/*has bullet*/ ListRE.test(getLineSafely(cm, lineNo))) {
            lineNo--
            break
          }
          addIndentTo[lineNo] = spaces
        }
      }
      if (!rangeEmpty) break
    } else {
      cm.execCommand('indentMore')
    }
  }
  cm.operation(() => {
    // if (!(flag0 || flag1 || flag2)) return cm.execCommand("defaultTab")
    // console.log(flag0, flag1, flag2)
    for (const lineNo in addIndentTo) {
      if (addIndentTo[lineNo])
        cm.replaceRange(addIndentTo[lineNo], { line: ~~lineNo, ch: 0 })
    }
    if (flag0) cm.setSelections(selections)
    if (flag1) cm.replaceSelections(beforeCur)
    if (flag2) cm.replaceSelections(afterCur, 'start')
  })
}

Object.assign(CodeMirror.commands, {
  enhancedShiftTab: shiftTab,
  enhancedTab: tab
})
