// HyperMD, copyright (c) by laobubu
// Distributed under an MIT license: http://laobubu.net/HyperMD/LICENSE
//
// powerful keymap for HyperMD and Markdown modes
//
import * as CodeMirror from 'codemirror'
import { cmpPos } from 'codemirror'
import repeat from 'repeat-string'
import { TableType } from './mode-gfm'
import { TokenSeeker, getGfmState } from './utils'

const isRealTableSep = token =>
  /table-sep/.test(token.type) && !/table-sep-dummy/.test(token.type)

CodeMirror.commands.newTableRowAndContinue = cm => {
  if (cm.getOption('disableInput')) return CodeMirror.Pass
  const selections = cm.listSelections()
  for (const range of selections) {
    if (newTableRowAndContinue(cm, range)) {
      return
    }
  }
}

/**
 * continue list / quote / insert table row
 * start a table
 */
export function newTableRowAndContinue(cm, range) {
  const pos = range.head
  const rangeEmpty = range.empty()
  const eolState = getGfmState(cm.getStateAfter(pos.line))
  const line = cm.getLine(pos.line)

  const table = rangeEmpty ? eolState.table : TableType.NONE
  if (table !== TableType.NONE) {
    if (
      /^[\s\|]+$/.test(line) &&
      (pos.line === cm.lastLine() ||
        getGfmState(cm.getStateAfter(pos.line + 1)).table !== table)
    ) {
      // if this is last row and is empty
      // remove this row and insert a new line
      cm.setCursor({ line: pos.line, ch: 0 })
      cm.replaceRange(
        '\n',
        { line: pos.line, ch: 0 },
        { line: pos.line, ch: line.length }
      )
    } else {
      // insert a row below
      const columns = eolState.tableColumns
      let newline = repeat(' | ', columns.length - 1)
      let leading = '\n'
      if (table === TableType.NORMAL) {
        leading += '| '
        newline += ' |'
      }
      // There are always nut users!
      if (eolState.tableRow == 0) {
        cm.setCursor({
          line: pos.line + 1,
          ch: cm.getLine(pos.line + 1).length
        })
      } else {
        cm.setCursor({ line: pos.line, ch: line.length })
      }
      cm.replaceSelection(leading)
      cm.replaceSelection(newline, 'start')
    }
    return true
  } else if (
    rangeEmpty &&
    pos.ch >= line.length &&
    !eolState.code &&
    !eolState.innerMode &&
    /^\|.+\|.+\|$/.test(line)
  ) {
    // current line is   | this | format |
    // let's make a table
    const lineTokens = cm.getLineTokens(pos.line)
    let ans = '|',
      ans2 = '|'
    for (let i = 1; i < lineTokens.length; i++) {
      // first token must be "|"
      const token = lineTokens[i]
      if (token.string === '|' && (!token.type || !token.type.trim().length)) {
        ans += ' ------- |'
        ans2 += ' | '
      }
    }
    // multi-cursor is meanless for this
    // replacements.push("\n" + ans + "\n" + ans2 + "\n")
    cm.setCursor({ line: pos.line, ch: line.length })
    cm.replaceSelection('\n' + ans + '\n| ')
    cm.replaceSelection(ans2.trim().slice(1) + '\n', 'start')
    return true
  }

  return false
}

CodeMirror.commands.moveToNextTableCell = cm => {
  var selections = [...cm.listSelections()]
  var beforeCur = []
  var afterCur = []
  /** indicate previous 4 variable changed or not */
  var flag0 = false,
    flag1 = false,
    flag2 = false
  function setBeforeCur(i, text) {
    beforeCur[i] = text
    if (text) flag1 = true
  }
  function setAfterCur(i, text) {
    afterCur[i] = text
    if (text) flag2 = true
  }
  for (let i = 0; i < selections.length; i++) {
    beforeCur[i] = afterCur[i] = ''
    var range = selections[i]
    var left = Object.assign({}, range.head)
    const eolState = getGfmState(cm.getStateAfter(left.line))
    if (eolState.table) {
      // yeah, we are inside a table
      flag0 = true // cursor will move
      beforeCur[i] = afterCur[i] = ''
      const result = moveToNextTableCell(cm, range)
      result.beforeCur && setBeforeCur(i, result.beforeCur)
      result.afterCur && setAfterCur(i, result.afterCur)
      if (result.moveTo) {
        selections[i] = result.moveTo
      }
    }
  }
  cm.operation(() => {
    if (flag0) cm.setSelections(selections)
    if (flag1) cm.replaceSelections(beforeCur)
    if (flag2) cm.replaceSelections(afterCur, 'start')
  })
}

export function moveToNextTableCell(cm, range) {
  var beforeCur = undefined
  var afterCur = undefined
  var moveTo = undefined
  const tokenSeeker = new TokenSeeker(cm)

  var left = { ...range.head }
  var right = { ...range.anchor }

  const rangeEmpty = range.empty()
  if (!rangeEmpty && cmpPos(left, right) > 0) [right, left] = [left, right]
  else if (right === left) {
    right = range.anchor = { ch: left.ch, line: left.line }
  }
  const eolState = getGfmState(cm.getStateAfter(left.line))
  const line = cm.getLine(left.line)
  if (eolState.table) {
    // yeah, we are inside a table
    const isNormalTable = eolState.table === TableType.NORMAL
    const columns = eolState.tableColumns
    tokenSeeker.setPos(left.line, left.ch)
    const nextCellLeft = tokenSeeker.findNext(
      isRealTableSep,
      tokenSeeker.i_token
    )
    if (!nextCellLeft) {
      // already last cell
      const lineSpan = eolState.tableRow === 0 ? 2 : 1 // skip |---|---| line
      if (
        left.line + lineSpan > cm.lastLine() ||
        getGfmState(cm.getStateAfter(left.line + lineSpan)).table !=
          eolState.table
      ) {
        // insert a row after this line
        left.ch = right.ch = line.length
        const newline = repeat(' | ', columns.length - 1)
        // There are always nut users!
        if (eolState.tableRow === 0) {
          right.line = left.line += 1
          right.ch = left.ch = cm.getLine(left.line).length
        }
        if (isNormalTable) {
          beforeCur = '\n| '
          afterCur = newline + ' |'
        } else {
          beforeCur = '\n'
          afterCur = newline.trimRight()
        }
        moveTo = { head: right, anchor: left }
      } else {
        // move cursor to next line, first cell
        right.line = left.line += lineSpan
        tokenSeeker.setPos(left.line, 0)
        const line = tokenSeeker.line.text
        const dummySep =
          isNormalTable && tokenSeeker.findNext(/table-sep-dummy/, 0)
        const nextCellRight = tokenSeeker.findNext(
          /table-sep/,
          dummySep ? dummySep.i_token + 1 : 1
        )
        left.ch = dummySep ? dummySep.token.end : 0
        right.ch = nextCellRight ? nextCellRight.token.start : line.length
        if (right.ch > left.ch && line.charAt(left.ch) === ' ') left.ch++
        if (right.ch > left.ch && line.charAt(right.ch - 1) === ' ') right.ch--
        moveTo = { head: right, anchor: left }
      }
    } else {
      const nextCellRight = tokenSeeker.findNext(
        /table-sep/,
        nextCellLeft.i_token + 1
      )
      left.ch = nextCellLeft.token.end
      right.ch = nextCellRight ? nextCellRight.token.start : line.length
      if (right.ch > left.ch && line.charAt(left.ch) === ' ') left.ch++
      if (right.ch > left.ch && line.charAt(right.ch - 1) === ' ') right.ch--
      moveTo = { head: right, anchor: left }
    }
  }
  return { moveTo, beforeCur, afterCur }
}

CodeMirror.commands.moveToPreviousTableCell = cm => {
  const selections = cm.listSelections()
  for (let i = 0; i < selections.length; i++) {
    const range = selections[i]
    const sel = moveToPreviousTableCell(cm, range)
    if (sel) {
      cm.setSelection(sel.anchor, sel.head)
      return
    }
  }
}

export function moveToPreviousTableCell(cm, range) {
  const tokenSeeker = new TokenSeeker(cm)
  const left = { ...range.head }
  const eolState = getGfmState(cm.getStateAfter(left.line))

  if (eolState.table) {
    tokenSeeker.setPos(left.line, left.ch)
    const isNormalTable = eolState.table === TableType.NORMAL // leading and ending | is not omitted
    var line = left.line
    var lineText = cm.getLine(line)
    var chStart = 0,
      chEnd = 0
    var rightPipe = tokenSeeker.findPrev(isRealTableSep)
    if (rightPipe) {
      // prev cell is in this line
      var leftPipe = tokenSeeker.findPrev(isRealTableSep, rightPipe.i_token - 1)
      chStart = leftPipe ? leftPipe.token.end : 0
      chEnd = rightPipe.token.start
      if (chStart == 0 && isNormalTable)
        chStart += lineText.match(/^\s*\|/)[0].length
    } else {
      // jump to prev line, last cell
      if (eolState.tableRow == 0) return // no more row before
      if (eolState.tableRow == 2) line-- // skip row #1 (| ----- | ----- |)
      line--
      lineText = cm.getLine(line)
      tokenSeeker.setPos(line, lineText.length)
      var leftPipe = tokenSeeker.findPrev(isRealTableSep)
      chStart = leftPipe.token.end
      chEnd = lineText.length
      if (isNormalTable) chEnd -= lineText.match(/\|\s*$/)[0].length
    }
    if (lineText.charAt(chStart) === ' ') chStart += 1
    if (chStart > 0 && lineText.substr(chStart - 1, 2) === ' |') chStart--
    if (lineText.charAt(chEnd - 1) === ' ') chEnd -= 1
    return {
      anchor: { line, ch: chStart },
      head: { line, ch: chEnd }
    }
  } else {
    return false
  }
}
