/**
 * CodeMirror-related utils
 *
 * @internal Part of HyperMD core.
 *
 * You shall NOT import this file; please import "core" instead
 */
/**
 * Useful tool to seek for tokens
 *
 *     var seeker = new TokenSeeker(cm)
 *     seeker.setPos(0, 0) // set to line 0, char 0
 *     var ans = seeker.findNext(/fomratting-em/)
 *
 */
export class TokenSeeker {
  constructor(cm) {
    this.cm = cm
  }
  findNext(condition, varg, since) {
    var lineNo = this.lineNo
    var tokens = this.lineTokens
    var token = null
    var i_token = this.i_token + 1
    var maySpanLines = false
    if (varg === true) {
      maySpanLines = true
    } else if (typeof varg === 'number') {
      i_token = varg
    }
    if (since) {
      if (since.line > lineNo) {
        i_token = tokens.length // just ignore current line
      } else if (since.line < lineNo) {
        // hmmm... we shall NEVER go back
      } else {
        for (; i_token < tokens.length; i_token++) {
          if (tokens[i_token].start >= since.ch) break
        }
      }
    }
    for (; i_token < tokens.length; i_token++) {
      var token_tmp = tokens[i_token]
      if (
        typeof condition === 'function'
          ? condition(token_tmp)
          : condition.test(token_tmp.type)
      ) {
        token = token_tmp
        break
      }
    }
    if (!token && maySpanLines) {
      const cm = this.cm
      const startLine = Math.max(since ? since.line : 0, lineNo + 1)
      cm.eachLine(startLine, cm.lastLine() + 1, line_i => {
        lineNo = line_i.lineNo()
        tokens = cm.getLineTokens(lineNo)
        i_token = 0
        if (since && lineNo === since.line) {
          for (; i_token < tokens.length; i_token++) {
            if (tokens[i_token].start >= since.ch) break
          }
        }
        for (; i_token < tokens.length; i_token++) {
          var token_tmp = tokens[i_token]
          if (
            typeof condition === 'function'
              ? condition(token_tmp)
              : condition.test(token_tmp.type)
          ) {
            token = token_tmp
            return true // stop `eachLine`
          }
        }
      })
    }
    return token ? { lineNo, token, i_token } : null
  }
  findPrev(condition, varg, since) {
    var lineNo = this.lineNo
    var tokens = this.lineTokens
    var token = null
    var i_token = this.i_token - 1
    var maySpanLines = false
    if (varg === true) {
      maySpanLines = true
    } else if (typeof varg === 'number') {
      i_token = varg
    }
    if (since) {
      if (since.line < lineNo) {
        i_token = -1 // just ignore current line
      } else if (since.line > lineNo) {
        // hmmm... we shall NEVER go forward
      } else {
        for (; i_token < tokens.length; i_token++) {
          if (tokens[i_token].start >= since.ch) break
        }
      }
    }
    if (i_token >= tokens.length) i_token = tokens.length - 1
    for (; i_token >= 0; i_token--) {
      var token_tmp = tokens[i_token]
      if (
        typeof condition === 'function'
          ? condition(token_tmp)
          : condition.test(token_tmp.type)
      ) {
        token = token_tmp
        break
      }
    }
    if (!token && maySpanLines) {
      const cm = this.cm
      const startLine = Math.min(since ? since.line : cm.lastLine(), lineNo - 1)
      const endLine = cm.firstLine()
      // cm.eachLine doesn't support reversed searching
      // use while... loop to iterate
      lineNo = startLine + 1
      while (!token && endLine <= --lineNo) {
        const _line_i = cm.getLineHandle(lineNo)
        tokens = cm.getLineTokens(lineNo)
        i_token = 0
        if (since && lineNo === since.line) {
          for (; i_token < tokens.length; i_token++) {
            if (tokens[i_token].start >= since.ch) break
          }
        }
        if (i_token >= tokens.length) i_token = tokens.length - 1
        for (; i_token >= 0; i_token--) {
          var token_tmp = tokens[i_token]
          if (
            typeof condition === 'function'
              ? condition(token_tmp)
              : condition.test(token_tmp.type)
          ) {
            token = token_tmp
            break // FOUND token !
          }
        }
      }
    }
    return token ? { lineNo, token, i_token } : null
  }
  /**
   * return a range in which every token has the same style, or meet same condition
   */
  expandRange(style, maySpanLines) {
    const cm = this.cm
    var isStyled
    if (typeof style === 'function') {
      isStyled = style
    } else {
      if (typeof style === 'string')
        style = new RegExp('(?:^|\\s)' + style + '(?:\\s|$)')
      isStyled = token => (token ? style.test(token.type || '') : false)
    }
    var from = {
      lineNo: this.lineNo,
      i_token: this.i_token,
      token: this.lineTokens[this.i_token]
    }
    var to = Object.assign({}, from)
    // find left
    var foundUnstyled = false,
      tokens = this.lineTokens,
      i = this.i_token
    while (!foundUnstyled) {
      if (i >= tokens.length) i = tokens.length - 1
      for (; i >= 0; i--) {
        const token = tokens[i]
        if (!isStyled(token)) {
          foundUnstyled = true
          break
        } else {
          from.i_token = i
          from.token = token
        }
      }
      if (foundUnstyled || !(maySpanLines && from.lineNo > cm.firstLine()))
        break // found, or no more lines
      tokens = cm.getLineTokens(--from.lineNo)
      i = tokens.length - 1
    }
    // find right
    var foundUnstyled = false,
      tokens = this.lineTokens,
      i = this.i_token
    while (!foundUnstyled) {
      if (i < 0) i = 0
      for (; i < tokens.length; i++) {
        const token = tokens[i]
        if (!isStyled(token)) {
          foundUnstyled = true
          break
        } else {
          to.i_token = i
          to.token = token
        }
      }
      if (foundUnstyled || !(maySpanLines && to.lineNo < cm.lastLine())) break // found, or no more lines
      tokens = cm.getLineTokens(++to.lineNo)
      i = 0
    }
    return { from, to }
  }
  setPos(line, ch, precise) {
    if (ch === void 0) {
      ch = line
      line = this.line
    } else if (typeof line === 'number') line = this.cm.getLineHandle(line)
    const sameLine = line === this.line
    var i_token = 0
    if (precise || !sameLine) {
      this.line = line
      this.lineNo = line.lineNo()
      this.lineTokens = this.cm.getLineTokens(this.lineNo)
    } else {
      // try to speed-up seeking
      i_token = this.i_token
      const token = this.lineTokens[i_token]
      if (token.start > ch) i_token = 0
    }
    var tokens = this.lineTokens
    for (; i_token < tokens.length; i_token++) {
      if (tokens[i_token].end > ch) break // found
    }
    this.i_token = i_token
  }
  /** get (current or idx-th) token */
  getToken(idx) {
    if (typeof idx !== 'number') idx = this.i_token
    return this.lineTokens[idx]
  }
  /** get (current or idx-th) token type. always return a string */
  getTokenType(idx) {
    if (typeof idx !== 'number') idx = this.i_token
    var t = this.lineTokens[idx]
    return (t && t.type) || ''
  }
}
/**
 * CodeMirror's `getLineTokens` might merge adjacent chars with same styles,
 * but this one won't.
 *
 * This one will consume more memory.
 *
 * @param {CodeMirror.LineHandle} line
 * @returns {string[]} every char's style
 */
export function getEveryCharToken(line) {
  var ans = new Array(line.text.length)
  var ss = line.styles
  var i = 0
  if (ss) {
    // CodeMirror already parsed this line. Use cache
    for (let j = 1; j < ss.length; j += 2) {
      const i_to = ss[j],
        s = ss[j + 1]
      while (i < i_to) ans[i++] = s
    }
  } else {
    // Emmm... slow method
    const cm =
      line.parent.cm || line.parent.parent.cm || line.parent.parent.parent.cm
    const ss = cm.getLineTokens(line.lineNo())
    for (let j = 0; j < ss.length; j++) {
      const i_to = ss[j].end,
        s = ss[j].type
      while (i < i_to) ans[i++] = s
    }
  }
  return ans
}
/**
 * return a range in which every char has the given style (aka. token type).
 * assuming char at `pos` already has the style.
 *
 * the result will NOT span lines.
 *
 * @param style aka. token type
 * @see TokenSeeker if you want to span lines
 */
export function expandRange(cm, pos, style) {
  var line = pos.line
  var from = { line, ch: 0 }
  var to = { line, ch: pos.ch }
  var styleFn = typeof style === 'function' ? style : false
  var styleRE = !styleFn && new RegExp('(?:^|\\s)' + style + '(?:\\s|$)')
  var tokens = cm.getLineTokens(line)
  var iSince
  for (iSince = 0; iSince < tokens.length; iSince++) {
    if (tokens[iSince].end >= pos.ch) break
  }
  if (iSince === tokens.length) return null
  for (var i = iSince; i < tokens.length; i++) {
    var token = tokens[i]
    if (styleFn ? styleFn(token) : styleRE.test(token.type)) to.ch = token.end
    else break
  }
  for (var i = iSince; i >= 0; i--) {
    var token = tokens[i]
    if (!(styleFn ? styleFn(token) : styleRE.test(token.type))) {
      from.ch = token.end
      break
    }
  }
  return { from, to }
}
/**
 * clean line measure caches (if needed)
 * and re-position cursor
 *
 * partially extracted from codemirror.js : function updateSelection(cm)
 *
 * @param {cm_t} cm
 * @param {boolean} skipCacheCleaning
 */
export function updateCursorDisplay(cm, skipCacheCleaning) {
  if (!skipCacheCleaning) {
    var lvs = cm.display.view // LineView s
    for (var lineView of lvs) {
      if (lineView.measure) lineView.measure.cache = {}
    }
  }
  setTimeout(function () {
    cm.display.input.showSelection(cm.display.input.prepareSelection())
  }, 60) // wait for css style
}

export function getGfmState(state) {
  function getBaseState(state) {
    return state.overlay && state.base ? state.base : undefined
  }

  if (state.hasOwnProperty('fencedEndRE')) {
    return state
  } else {
    const baseState = getBaseState(state)
    return baseState ? getGfmState(baseState) : state
  }
}
